// Get the browser API (Chrome or Firefox)
const browserAPI = typeof browser !== 'undefined' ? browser : chrome;

// DOM elements
const form = document.getElementById('optionsForm');
const apiUrlInput = document.getElementById('apiUrl');
const saveButton = document.getElementById('saveButton');
const testButton = document.getElementById('testButton');
const statusMessage = document.getElementById('statusMessage');

// Load saved preferences
async function loadPreferences() {
  try {
    const result = await new Promise((resolve) => {
      browserAPI.storage.sync.get(['restApiUrl'], resolve);
    });
    
    if (result.restApiUrl) {
      apiUrlInput.value = result.restApiUrl;
    }
    
    // Load and display current keyboard shortcut
    await loadKeyboardShortcut();
    
    // Load and display categories
    await loadCategories();
  } catch (error) {
    console.error('Error loading preferences:', error);
    showStatus('Error loading preferences', 'error');
  }
}

// Load and display current keyboard shortcut
async function loadKeyboardShortcut() {
  try {
    const commands = await new Promise((resolve) => {
      if (browserAPI.commands && browserAPI.commands.getAll) {
        browserAPI.commands.getAll(resolve);
      } else {
        resolve([]);
      }
    });
    
    const openPopupCommand = commands.find(cmd => cmd.name === 'open-popup');
    if (openPopupCommand && openPopupCommand.shortcut) {
      const shortcutText = openPopupCommand.shortcut;
      document.getElementById('currentShortcut').textContent = shortcutText;
      document.getElementById('shortcutText').textContent = shortcutText;
      document.getElementById('shortcutTextMac').textContent = shortcutText.replace('Ctrl', 'Cmd');
      document.querySelector('.shortcut-hint').textContent = '(Current)';
    } else {
      // Default shortcut if not customized
      document.getElementById('currentShortcut').textContent = 'Ctrl+Shift+B';
      document.getElementById('shortcutText').textContent = 'Ctrl+Shift+B';
      document.getElementById('shortcutTextMac').textContent = 'Cmd+Shift+B';
      document.querySelector('.shortcut-hint').textContent = '(Default)';
    }
  } catch (error) {
    console.error('Error loading keyboard shortcut:', error);
    // Keep default values on error
  }
}

// Save preferences
async function savePreferences() {
  const apiUrl = apiUrlInput.value.trim();
  
  if (!apiUrl) {
    showStatus('Please enter a valid API URL', 'error');
    return;
  }
  
  try {
    await new Promise((resolve) => {
      browserAPI.storage.sync.set({ restApiUrl: apiUrl }, resolve);
    });
    
    showStatus('Settings saved successfully!', 'success');
  } catch (error) {
    console.error('Error saving preferences:', error);
    showStatus('Error saving preferences', 'error');
  }
}

// Test API connection
async function testConnection() {
  const apiUrl = apiUrlInput.value.trim();
  
  if (!apiUrl) {
    showStatus('Please enter an API URL first', 'error');
    return;
  }
  
  testButton.disabled = true;
  testButton.textContent = 'Testing...';
  showStatus('Testing connection...', 'info');
  
  try {
    // Try to make a simple request to test the connection
    const response = await fetch(apiUrl, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json'
      }
    });
    
    if (response.ok || response.status === 404 || response.status === 405) {
      // 404/405 might be expected if endpoint doesn't exist, but connection works
      showStatus('Connection successful!', 'success');
    } else {
      showStatus(`Connection failed: ${response.status} ${response.statusText}`, 'error');
    }
  } catch (error) {
    showStatus(`Connection failed: ${error.message}`, 'error');
  } finally {
    testButton.disabled = false;
    testButton.textContent = 'Test Connection';
  }
}

// Show status message
function showStatus(message, type = 'info') {
  statusMessage.textContent = message;
  statusMessage.className = `status-message ${type}`;
  statusMessage.style.display = 'block';
  
  // Auto-hide success messages after 3 seconds
  if (type === 'success') {
    setTimeout(() => {
      statusMessage.style.display = 'none';
    }, 3000);
  }
}

// Category management
let categories = {};

// Load categories from storage
async function loadCategories() {
  try {
    const result = await new Promise((resolve) => {
      browserAPI.storage.sync.get(['categories'], resolve);
    });
    categories = result.categories || {};
    displayCategories();
  } catch (error) {
    console.error('Error loading categories:', error);
    categories = {};
    displayCategories();
  }
}

// Save categories to storage
async function saveCategories() {
  try {
    await new Promise((resolve) => {
      browserAPI.storage.sync.set({ categories: categories }, resolve);
    });
    showStatus('Categories saved successfully!', 'success');
  } catch (error) {
    console.error('Error saving categories:', error);
    showStatus('Error saving categories', 'error');
  }
}

// Display categories in UI
function displayCategories() {
  const container = document.getElementById('categoriesContainer');
  container.innerHTML = '';
  
  const categoryNames = Object.keys(categories);
  
  if (categoryNames.length === 0) {
    container.innerHTML = '<p style="color: #6a6a6a; font-size: 12px;">No categories yet. Click "Add Category" to create one.</p>';
    return;
  }
  
  categoryNames.forEach((categoryName, index) => {
    const categoryDiv = document.createElement('div');
    categoryDiv.className = 'category-item';
    categoryDiv.style.cssText = 'margin-bottom: 20px; padding: 12px; border: 1px solid #b5d5e8; background: #faf9f6;';
    
    const headerDiv = document.createElement('div');
    headerDiv.style.cssText = 'display: flex; align-items: center; gap: 8px; margin-bottom: 8px;';
    
    const nameInput = document.createElement('input');
    nameInput.type = 'text';
    nameInput.value = categoryName;
    nameInput.style.cssText = 'flex: 1; padding: 6px 8px; border: 1px solid #b5d5e8; font-size: 14px; background: #faf9f6; color: #4a4a4a;';
    nameInput.addEventListener('change', () => {
      const newName = nameInput.value.trim();
      if (newName && newName !== categoryName) {
        categories[newName] = categories[categoryName];
        delete categories[categoryName];
        saveCategories();
        loadCategories();
      }
    });
    
    const deleteButton = document.createElement('button');
    deleteButton.type = 'button';
    deleteButton.textContent = 'Delete';
    deleteButton.style.cssText = 'padding: 6px 12px; background: #e8b5d5; color: #4a4a4a; border: 1px solid #e8b5d5; cursor: pointer; font-size: 12px;';
    deleteButton.addEventListener('click', () => {
      if (confirm(`Delete category "${categoryName}"?`)) {
        delete categories[categoryName];
        saveCategories();
        loadCategories();
      }
    });
    
    headerDiv.appendChild(nameInput);
    headerDiv.appendChild(deleteButton);
    
    const domainsDiv = document.createElement('div');
    domainsDiv.style.cssText = 'margin-top: 8px;';
    
    const domains = categories[categoryName] || [];
    domains.forEach(domain => {
      const domainDiv = document.createElement('div');
      domainDiv.style.cssText = 'display: flex; align-items: center; gap: 8px; margin-bottom: 4px; padding: 4px 8px; background: #fff; border: 1px solid #d0c9c0;';
      
      const domainSpan = document.createElement('span');
      domainSpan.textContent = domain;
      domainSpan.style.cssText = 'flex: 1; font-size: 12px; color: #4a4a4a;';
      
      const removeButton = document.createElement('button');
      removeButton.type = 'button';
      removeButton.textContent = '×';
      removeButton.style.cssText = 'width: 20px; height: 20px; padding: 0; background: #f4d5a8; color: #4a4a4a; border: 1px solid #f4d5a8; cursor: pointer; font-size: 16px; line-height: 1; display: flex; align-items: center; justify-content: center;';
      removeButton.addEventListener('click', () => {
        const index = categories[categoryName].indexOf(domain);
        if (index > -1) {
          categories[categoryName].splice(index, 1);
          if (categories[categoryName].length === 0) {
            delete categories[categoryName];
          }
          saveCategories();
          loadCategories();
        }
      });
      
      domainDiv.appendChild(domainSpan);
      domainDiv.appendChild(removeButton);
      domainsDiv.appendChild(domainDiv);
    });
    
    const addDomainDiv = document.createElement('div');
    addDomainDiv.style.cssText = 'display: flex; gap: 8px; margin-top: 8px;';
    
    const domainInput = document.createElement('input');
    domainInput.type = 'text';
    domainInput.placeholder = 'Add domain (e.g., example.com)';
    domainInput.style.cssText = 'flex: 1; padding: 6px 8px; border: 1px solid #b5d5e8; font-size: 12px; background: #faf9f6; color: #4a4a4a;';
    
    const addDomainButton = document.createElement('button');
    addDomainButton.type = 'button';
    addDomainButton.textContent = 'Add';
    addDomainButton.style.cssText = 'padding: 6px 12px; background: #d5e8b5; color: #4a4a4a; border: 1px solid #d5e8b5; cursor: pointer; font-size: 12px;';
    addDomainButton.addEventListener('click', () => {
      const domain = domainInput.value.trim();
      if (domain) {
        if (!categories[categoryName]) {
          categories[categoryName] = [];
        }
        if (!categories[categoryName].includes(domain)) {
          categories[categoryName].push(domain);
          saveCategories();
          loadCategories();
        }
        domainInput.value = '';
      }
    });
    
    domainInput.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') {
        addDomainButton.click();
      }
    });
    
    addDomainDiv.appendChild(domainInput);
    addDomainDiv.appendChild(addDomainButton);
    
    categoryDiv.appendChild(headerDiv);
    categoryDiv.appendChild(domainsDiv);
    categoryDiv.appendChild(addDomainDiv);
    
    container.appendChild(categoryDiv);
  });
}

// Add new category
function addNewCategory() {
  const categoryName = prompt('Enter category name:');
  if (categoryName && categoryName.trim()) {
    const trimmedName = categoryName.trim();
    if (!categories[trimmedName]) {
      categories[trimmedName] = [];
      saveCategories();
      loadCategories();
    } else {
      showStatus('Category already exists', 'error');
    }
  }
}

// Event listeners
form.addEventListener('submit', async (e) => {
  e.preventDefault();
  await savePreferences();
});

testButton.addEventListener('click', testConnection);

// Load preferences when page loads
document.addEventListener('DOMContentLoaded', () => {
  loadPreferences();
  
  // Attach add category button listener after DOM is ready
  const addCategoryButton = document.getElementById('addCategoryButton');
  if (addCategoryButton) {
    addCategoryButton.addEventListener('click', addNewCategory);
  }
});
